/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;

namespace Inet.Viewer.Data
{


    /// <summary>
    /// A Range is an immutable range of values between two limit values. The two limit values can be included in the range, but do not have to be.
    /// The value types of the objects set should be:
    /// <br/>String for STRING
    /// <br/>Double for NUMBER and CURRENCY
    /// <br/>Boolean for BOOLEAN
    /// <br/>Date for DATE and DATETIME
    /// <br/>Time for TIME
    /// <br/>byte[] for BINARY
    /// </summary>
    [Serializable]
    public class Range
    {

        private object lower;

        private object upper;

        private bool includeLower;

        private bool includeUpper;

        /// <summary>
        /// Creates a new range with the given lower and upper limits
        /// The value types of the objects set should be:
        /// <br/>String for STRING
        /// <br/>Double for NUMBER and CURRENCY
        /// <br/>Boolean for BOOLEAN
        /// <br/>Date for DATE and DATETIME
        /// <br/>Time for TIME
        /// <br/>byte[] for BINARY </summary>
        /// <param name="lower"> lower limit of the range </param>
        /// <param name="upper"> upper limit of the range </param>
        /// <param name="includeLower"> whether to include the lower limit in the range </param>
        /// <param name="includeUpper"> whether to include the upper limit in the range
        /// @since 7.8 </param>
        public Range(object lower, object upper, bool includeLower, bool includeUpper)
        {
            if (!(lower is PromptValue))
            {
                int valueType = ExtractValueType(lower);
                if (valueType == PromptData.Date)
                {
                    lower = ((DateTime)lower).Date;
                }
                lower = new SinglePromptValue(lower, string.Empty, valueType);
            }
            if (!(upper is PromptValue))
            {
                int valueType = ExtractValueType(lower);
                if (valueType == PromptData.Date)
                {
                    upper = ((DateTime)upper).Date;
                }
                upper = new SinglePromptValue(upper, string.Empty, valueType);
            }
            this.lower = lower;
            this.upper = upper;
            this.includeLower = includeLower;
            this.includeUpper = includeUpper;
        }

        /// <summary>
        /// extracts the value type of the object </summary>
        /// <param name="obj">  object to get value type for </param>
        /// <returns> prompt value type </returns>
        private int ExtractValueType(object obj)
        {
            if (obj is string)
            {
                return PromptData.String;
            }
            else if (obj is double?)
            {
                return PromptData.Number;
            }
            else if (obj is DateTime)
            {
                return PromptData.Date;
            }
            else if (obj is TimeSpan)
            {
                return PromptData.Time;
            }
            else if (obj is sbyte[])
            {
                return PromptData.Binary;
            }
            return -1;
        }

        /// <summary>
        /// Returns whether the upper limit is included. </summary>
        /// <returns> Is the upper limit included in the range?
        /// @since 7.8 </returns>
        internal bool IncludeUpper
        {
            get
            {
                return includeUpper;
            }
            set
            {
                this.includeUpper = value;
            }
        }


        /// <summary>
        /// Returns whether the lower limit is included. </summary>
        /// <returns> Is the lower limit included in the range?
        /// @since 7.8 </returns>
        internal bool IncludeLower
        {
            get
            {
                return includeLower;
            }
            set
            {
                this.includeLower = value;
            }
        }


        /// <summary>
        /// Returns the upper limit </summary>
        /// <returns> Upper limit to the range, or null if there is no upper limit
        /// @since 7.8 </returns>
        internal object UpperValue
        {
            get
            {
                return upper;
            }
            set
            {
                this.upper = value;
            }
        }


        /// <summary>
        /// Returns the lower limit </summary>
        /// <returns> Lower limit to the range, or null if there is no lower limit
        /// @since 7.8 </returns>
        internal object LowerValue
        {
            get
            {
                return lower;
            }
            set
            {
                this.lower = value;
            }
        }


    }

}